//OHSAT GAMES MEGATILER TUTORIAL - Loading Levels - https://www.ohsat.com/tutorial/megatiler/megatiler-7/
//MEGATEAMWORK Makes the MEGADREAM Work 
//SGDK Version: 2.0 

//https://www.ohsat.com/tutorial/#mega-drive-tutorials 

#include <genesis.h>
#include <resources.h>

//tile measurement defines
#define TILESIZE 8
#define MAP_WIDTH 8
#define MAP_HEIGHT 8

//array defines for tile usage 
#define SOLID_TILE 1
#define SPAWN_TILE 4
#define EXIT_TILE 5
#define COIN_TILE 6

//level variables
#define MAX_COINS 3
#define LEVEL_NUM 3

//Sprite animation defines
#define ANIM_DOWN 0
#define ANIM_UP 1
#define ANIM_SIDE 2

//defines for SFX
#define SFX_COIN 64
#define SFX_UNLOCK 65

// -------------------------
// STRUCT & TYPE DEFINITIONS
// -------------------------

typedef u8 map[MAP_HEIGHT][MAP_WIDTH];

typedef struct {
    u8 x;
    u8 y;
} Point;

typedef enum { up, down, left, right, none } moveDirection;

typedef struct {
    Point pos;
    Point tilePos;
    int w;
    int h;
    int health;
    bool moving;
    moveDirection dir;
    Sprite *sprite;
    char name[6];
} Entity;

typedef struct {
    Point pos;
    u8 w;
    u8 h;
    Sprite *sprite;
    u8 health;
} Coin;

// -------------------------
// LEVEL DEFINITIONS
// -------------------------

map level1 = {
    {6,0,0,0,0,0,0,6},
    {0,0,0,0,0,0,0,0},
    {0,0,0,0,1,0,0,0},
    {0,0,0,1,1,0,0,0},
    {4,0,0,0,1,0,0,0},
    {0,0,0,0,1,0,0,0},
    {0,0,0,0,1,0,0,0},
    {5,0,0,0,0,0,0,6}
};

map level2 = {
    {0,0,0,6,0,0,0,0},
    {0,0,0,0,0,0,0,0},
    {0,4,0,1,1,1,0,0},
    {0,0,0,6,0,1,0,0},
    {0,0,0,1,1,1,6,0},
    {0,0,0,1,0,0,0,0},
    {0,0,0,1,1,1,5,0},
    {0,0,0,0,0,0,0,0}
};

map level3 = {
    {0,0,0,0,0,0,0,0},
    {0,0,1,1,1,1,0,0},
    {0,0,0,0,6,1,0,0},
    {0,0,0,1,1,1,6,0},
    {0,0,0,0,0,1,0,0},
    {0,6,1,1,1,1,0,0},
    {0,4,0,0,0,0,5,0},
    {0,0,0,0,0,0,0,0}
};

// -------------------------
// GLOBALS
// -------------------------

u8 x = 0, y = 0, t = 0, i = 0;
u8 coinNum = 0;
u8 coinsCollected = 0;

map* currentLevel;
map* levelList[LEVEL_NUM] = { &level1, &level2, &level3 };
static u8 currentLevelIndex = 0;

char hud_string[10];

Point exitLocation = {0,0};
bool exitUnlocked = FALSE;

Entity player = {{0,0},{0,0},8,8,0,FALSE,none,NULL,"PLAYER"};

Coin coins[MAX_COINS];
Coin *coinToCheck;

// -------------------------
// FUNCTION HEADERS
// -------------------------

void loadLevel();
void movePlayer(moveDirection Direction);
void myJoyHandler(u16 joy, u16 changed, u16 state);
void updateScoreDisplay();
void unlockExit();
void clearLevel();

int getTileAt(u8 X, u8 Y)
{
    return (*currentLevel)[Y][X];
}

// -------------------------
// MAIN LOOP
// -------------------------

int main()
{
    XGM_setPCM(SFX_COIN, sfx_coin, sizeof(sfx_coin));
    XGM_setPCM(SFX_UNLOCK, sfx_unlock, sizeof(sfx_unlock));

    JOY_init();
    JOY_setEventHandler(&myJoyHandler);

    loadLevel();
    updateScoreDisplay();

    while(1)
    {
        if (player.moving)
        {
            switch(player.dir)
            {
                case up:    player.pos.y -= 1; break;
                case down:  player.pos.y += 1; break;
                case left:  player.pos.x -= 1; break;
                case right: player.pos.x += 1; break;
                default: break;
            }
        }

        SPR_update();

        if (player.pos.x % TILESIZE == 0 && player.pos.y % TILESIZE == 0)
        {
            player.moving = FALSE;

            if (exitUnlocked &&
                player.tilePos.x == exitLocation.x &&
                player.tilePos.y == exitLocation.y)
            {
                currentLevelIndex++;
                if (currentLevelIndex >= LEVEL_NUM)
                    currentLevelIndex = 0;

                loadLevel();
            }
        }

        SPR_setPosition(player.sprite, player.pos.x, player.pos.y);

        // COIN CHECK
        for (i = 0; i < MAX_COINS; i++)
        {
            coinToCheck = &coins[i];

            if (coinToCheck->health &&
                player.pos.x < coinToCheck->pos.x + coinToCheck->w &&
                player.pos.x + player.w > coinToCheck->pos.x &&
                player.pos.y < coinToCheck->pos.y + coinToCheck->h &&
                player.pos.y + player.h > coinToCheck->pos.y)
            {
                coinToCheck->health = 0;
                SPR_setVisibility(coinToCheck->sprite, HIDDEN);

                coinsCollected++;
                XGM_startPlayPCM(SFX_COIN, 1, SOUND_PCM_CH2);

                updateScoreDisplay();

                if (coinsCollected == MAX_COINS)
                    unlockExit();
            }
        }

        SYS_doVBlankProcess();
    }

    return 0;
}

// -------------------------
// LEVEL LOADING
// -------------------------

void loadLevel()
{
    clearLevel();
    currentLevel = levelList[currentLevelIndex];
    SPR_init();

    x = 0;
    y = 0;
    coinNum = 0;

    // reset coins
    for (i = 0; i < MAX_COINS; i++)
        coins[i].health = 0;

    for (i = 0; i < MAP_WIDTH * MAP_HEIGHT; i++)
    {
        t = (*currentLevel)[y][x];

        if (t == SPAWN_TILE)
        {
            player.tilePos.x = x;
            player.tilePos.y = y;
            player.pos.x = x * TILESIZE;
            player.pos.y = y * TILESIZE;

            player.sprite = SPR_addSprite(&spr_player, player.pos.x, player.pos.y,
                                          TILE_ATTR(PAL2,0,FALSE,FALSE));

            VDP_setTileMapXY(BG_B, TILE_ATTR_FULL(PAL1,0,FALSE,FALSE, 1), x, y);
        }
        else if (t == COIN_TILE)
        {
            if (coinNum < MAX_COINS)
            {
                Coin *c = &coins[coinNum];
                c->pos.x = x * TILESIZE;
                c->pos.y = y * TILESIZE;
                c->w = 8;
                c->h = 8;
                c->health = 1;

                c->sprite = SPR_addSprite(&coin, c->pos.x, c->pos.y,
                                          TILE_ATTR(PAL2,0,FALSE,FALSE));

                coinNum++;

                VDP_setTileMapXY(BG_B, TILE_ATTR_FULL(PAL1,0,FALSE,FALSE, 1), x, y);
            }
            
        }
        else if (t == EXIT_TILE)
        {
            exitLocation.x = x;
            exitLocation.y = y;
        }

        VDP_setTileMapXY(BG_B, TILE_ATTR_FULL(PAL1,0,FALSE,FALSE,t+1), x, y);

        x++;
        if (x >= MAP_WIDTH)
        {
            x = 0;
            y++;
        }
    }

    VDP_loadTileSet(floortiles.tileset, 1, DMA);
    PAL_setPalette(PAL1, floortiles.palette->data, DMA);
    PAL_setPalette(PAL2, spr_player.palette->data, DMA);
}

// -------------------------
// MOVEMENT HANDLER
// -------------------------

void movePlayer(moveDirection Direction)
{
    if (!player.moving)
    {
        switch(Direction)
        {
            case up:
                if (player.tilePos.y > 0 &&
                    getTileAt(player.tilePos.x, player.tilePos.y - 1) != SOLID_TILE)
                {
                    player.tilePos.y--;
                    player.moving = TRUE;
                    player.dir = up;
                    SPR_setAnim(player.sprite, ANIM_UP);
                }
                break;

            case down:
                if (player.tilePos.y < MAP_HEIGHT - 1 &&
                    getTileAt(player.tilePos.x, player.tilePos.y + 1) != SOLID_TILE)
                {
                    player.tilePos.y++;
                    player.moving = TRUE;
                    player.dir = down;
                    SPR_setAnim(player.sprite, ANIM_DOWN);
                }
                break;

            case left:
                if (player.tilePos.x > 0 &&
                    getTileAt(player.tilePos.x - 1, player.tilePos.y) != SOLID_TILE)
                {
                    player.tilePos.x--;
                    player.moving = TRUE;
                    player.dir = left;
                    SPR_setAnim(player.sprite, ANIM_SIDE);
                    SPR_setHFlip(player.sprite, TRUE);
                }
                break;

            case right:
                if (player.tilePos.x < MAP_WIDTH - 1 &&
                    getTileAt(player.tilePos.x + 1, player.tilePos.y) != SOLID_TILE)
                {
                    player.tilePos.x++;
                    player.moving = TRUE;
                    player.dir = right;
                    SPR_setAnim(player.sprite, ANIM_SIDE);
                    SPR_setHFlip(player.sprite, FALSE);
                }
                break;

            default:
                break;
        }
    }
}

// -------------------------
// INPUT HANDLER
// -------------------------

void myJoyHandler(u16 joy, u16 changed, u16 state)
{
    if (joy == JOY_1)
    {
        if (state & BUTTON_UP) movePlayer(up);
        else if (state & BUTTON_DOWN) movePlayer(down);
        else if (state & BUTTON_LEFT) movePlayer(left);
        else if (state & BUTTON_RIGHT) movePlayer(right);
    }
}

// -------------------------
// HUD / EXIT
// -------------------------

void updateScoreDisplay()
{
    sprintf(hud_string, "SCORE: %d", coinsCollected);
    VDP_clearText(8, 0, 10);
    VDP_drawText(hud_string, 8, 0);
}

void unlockExit()
{
    exitUnlocked = TRUE;
    XGM_startPlayPCM(SFX_UNLOCK, 1, SOUND_PCM_CH2);

    VDP_setTileMapXY(BG_B,
        TILE_ATTR_FULL(PAL1,0,FALSE,FALSE,3),
        exitLocation.x, exitLocation.y);
}

// -------------------------
// LEVEL CLEAR
// -------------------------

void clearLevel()
{
    VDP_clearPlane(BG_B, TRUE);
    VDP_clearSprites();

    coinsCollected = 0;
    exitUnlocked = FALSE;
}


////////////////////NOTES////////////////////

/*

Since this is the final lesson in OHSAT tutorial I had asked ChatGPT to structure my code in an
easier to understand format. 

Also, the C-language is very particular about it's ordering. 

I'll paste below the part of the loadLevel() function that is being modified to use
clearLevel() and currentLevel = levelList[currentLevelIndex];

void loadLevel()
{
    clearLevel();
    currentLevel = levelList[currentLevelIndex];
    SPR_init();

    x = 0;
    y = 0;
    coinNum = 0;

    // reset coins
    for (i = 0; i < MAX_COINS; i++)
        coins[i].health = 0;

    for (i = 0; i < MAP_WIDTH * MAP_HEIGHT; i++)
    {
        t = (*currentLevel)[y][x];

        if (t == SPAWN_TILE)
        {
            player.tilePos.x = x;
            player.tilePos.y = y;
            player.pos.x = x * TILESIZE;
            player.pos.y = y * TILESIZE;

            player.sprite = SPR_addSprite(&spr_player, player.pos.x, player.pos.y,
                                          TILE_ATTR(PAL2,0,FALSE,FALSE));
        }

Note: I was having trouble getting my code to compile using the method Andrej used in his tutorial. 

Instead of using u8 total = MAP_HEIGHT * MAP_WIDTH; and then trying for(i =0; i < total; i++){
t = (currentLevel + i)}; and then adding additional code to increment x and y (see below) I did it a different
way. 

OHSAT code

u8 i = 0;
u8 total = MAP_HEIGHT * MAP_WIDTH;

for(i = 0; i < total; i++){
    t = (current level + i);
}

x++;
if(x >= MAP_WIDTH){
    y++;
    x = 0;
}

My code

I declared my variables again. 

x = 0;
y = 0;
coinNum = 0;

Because of the globals, it was causing the values in the array to shift once I cleared level 1. 
The values for x and y on map 1 were carrying over to map 2 becuase it was holding the value for these
variables from the previous map. 

I also have it where the grass tiles aren't drawn behind the coin tiles nor does it obfuscate the spawn and exit tiles. 

If you're wanting to see something that's more true to the original OHSAT lesson I did another version 
titled MEGATILER 07 RETRY. I suggest you check it out. 

Thanks Andrej!

Cheers everyone! 


*/

/////////EXPERIMENTATION IDEAS///////////////

/*

1. I'm thinking of making some simple assets to make this into a Chip's Challenge kind of game. 
2. I may expand the map size to better incorporate a second player. We'll see what happens. 
3. I think I'll start working on condensing down the main.c file in the next lesson. 


*/

///////////ERROR HANDLING////////////////////

/*



*/